const subject7Gutenberg = {};
let started = false;
const uuid = 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'
    .replace(/[xy]/g, function (c) {
        const r = Math.random() * 16 | 0,
            v = c === 'x' ? r : (r & 0x3 | 0x8);
        return v.toString(16);
    });

subject7Gutenberg.xpathEngine = (() => {
    const generateXPath = async (element, hint, settings) => {
        let absoluteXpath = getAbsoluteXPath(element);
        let logInfo = [];
        let errors = [];
        let patternId = null;
        let patternDate = null;
        let result = {
            xpath: absoluteXpath,
            hint: hint,
            logInfo: logInfo,
            errors: errors,
            patternId: patternId,
            patternDate: patternDate
        };

        const data = await chrome.storage.local.get('templates');
        let xpathInfo = templateXPathEngine.generateXpathByPatterns(absoluteXpath, null, hint, data.templates || []);
        logInfo = xpathInfo.logInfo;
        patternId = xpathInfo.patternId;
        patternDate = xpathInfo.patternDate;
        let error = xpathInfo.error;

        if (xpathInfo.error) {
            errors.push(error);

            if (!xpathInfo.error.includes(CONSTANTS.ERROR_MESSAGES.ERROR_FETCHING_TARGET_ELEMENT_BY_XPATH)) {
                xpathInfo = smartXPathEngine.generateSmartXpath(absoluteXpath, null, hint, settings);
                error = xpathInfo.error;

                if (xpathInfo.error) {
                    errors.push(error);
                }
            }
        }

        result.xpath = xpathInfo.xpath ? xpathInfo.xpath : absoluteXpath;
        result.hint = xpathInfo.hint;
        result.logInfo = logInfo;
        result.errors = errors;
        result.patternId = patternId;
        result.patternDate = patternDate;
        return result;
    };

    const getAbsoluteXPath = element => {
        const tagNameInLowerCase = subject7Gutenberg.locatorRecorder.getTagName(element);
        if (tagNameInLowerCase === CONSTANTS.TAGS.HTML) {
            return CONSTANTS.XPATH_HELPERS.NODE_DIVIDER +
                CONSTANTS.TAGS.HTML + CONSTANTS.XPATH_HELPERS.BRACKETS_PATTERN
                    .replace(CONSTANTS.REPLACE_STRINGS.VALUE, '1');
        }
        if (element === document.body) {
            return CONSTANTS.XPATH_HELPERS.NODE_DIVIDER +
                CONSTANTS.TAGS.HTML + CONSTANTS.XPATH_HELPERS.BRACKETS_PATTERN
                    .replace(CONSTANTS.REPLACE_STRINGS.VALUE, '1') +
                CONSTANTS.XPATH_HELPERS.NODE_DIVIDER +
                CONSTANTS.TAGS.BODY + CONSTANTS.XPATH_HELPERS.BRACKETS_PATTERN
                    .replace(CONSTANTS.REPLACE_STRINGS.VALUE, '1');
        }
        let index = 0;
        const siblings = element.parentNode.childNodes;
        for (let i = 0; i < siblings.length; i++) {
            let sibling = siblings[i];
            if (sibling === element) {
                return getAbsoluteXPath(element.parentNode) +
                    CONSTANTS.XPATH_HELPERS.NODE_DIVIDER +
                    ((subject7Gutenberg.locatorRecorder.isSVGElement(element) || subject7Gutenberg.locatorRecorder.containsCustomNamespace(element)) ? `*[name()='${tagNameInLowerCase}']` : tagNameInLowerCase) +
                    CONSTANTS.XPATH_HELPERS.BRACKETS_PATTERN
                        .replace(CONSTANTS.REPLACE_STRINGS.VALUE, (index + 1).toString());
            }
            if (sibling.nodeType === Node.ELEMENT_NODE &&
                subject7Gutenberg.locatorRecorder.getTagName(sibling) === tagNameInLowerCase) {
                index++;
            }
        }
    };

    const getElementByXpath = (xpath, targetDocument) => {
        let result = targetDocument.evaluate(xpath, targetDocument, null, XPathResult.ORDERED_NODE_ITERATOR_TYPE, null);
        if (result.invalidIteratorState) {
            throw {code: 2, message: CONSTANTS.ERROR_MESSAGES.INVALID_XPATH};
        }

        let elements = [];
        let element;
        while ((element = result.iterateNext()) !== null) {
            elements.push(element);
        }

        if (elements.length === 0) {
            throw {code: 1, message: CONSTANTS.ERROR_MESSAGES.ELEMENT_NOT_FOUND_XPATH};
        } else {
            if (elements.length === 1) {
                return elements[0];
            } else {
                throw {code: 0, message: CONSTANTS.ERROR_MESSAGES.XPATH_RETURNS_SEVERAL_ELEMENTS, elements: elements};
            }
        }
    };

    return {
        generateXPath: generateXPath,
        getElementByXpath: getElementByXpath,
    };

})();

subject7Gutenberg.cssSelectorEngine = (() => {

    const generateCssForElementInShadowDom = composedPath => {
        if (!composedPath) {
            throw new Error(CONSTANTS.ERROR_MESSAGES.CSS_CAN_NOT_BE_GENERATED);
        }
        let element = composedPath[0];
        let cssSelectorArr = [];
        let cssSelector = generateCssSelector(element, true);
        let shadowHosts = getShadowHosts(element);

        shadowHosts.forEach(host => {
            cssSelectorArr.push(host.concat(CONSTANTS.CSS.SHADOW_PSEUDO));
        });
        cssSelectorArr.push(cssSelector);

        return cssSelectorArr.join(' ');
    };

    const generateCssSelector = (element, isElementInShadowDom) => {
        const nodeName = getNodeName(element);
        const isSvgElement = subject7Gutenberg.locatorRecorder.isSVGElement(element);
        let cssSelector = nodeName;
        let foundedElements = null;

        try {
            if (element.id && isValidIdentifier(element.id)) {
                cssSelector = '#'.concat(element.id.trim());
                foundedElements = foundElementsByCssSelector(element, cssSelector, isElementInShadowDom);

                if (foundedElements) {
                    cssSelector = getClarifiedCssSelector(cssSelector, element, foundedElements);

                    return cssSelector.trim();
                }
            }

            if (isSvgElement) {
                cssSelector = nodeName;
                foundedElements = foundElementsByCssSelector(element, cssSelector, isElementInShadowDom);

                if (foundedElements) {
                    cssSelector = getClarifiedCssSelector(cssSelector, element, foundedElements);

                    return cssSelector.trim();
                }
            }

            if (element.attributes.length) {
                let name;
                let nodeValue;
                let attributesObj = {};

                Array.from(element.attributes).forEach(attribute => {
                    name = attribute.name;
                    nodeValue = attribute.nodeValue;

                    if (nodeValue || nodeValue !== 'id' || nodeValue !== 'style') {
                        attributesObj[name] = nodeValue;
                    }
                });

                let stop = false;
                CONSTANTS.HINT.ATTRIBUTES_LIST.forEach(attr => {
                    if (stop) {
                        return;
                    }

                    if (attr in attributesObj) {
                        cssSelector = getCssByAttribute(element, cssSelector, attributesObj, attr);
                        stop = true;
                    } else {
                        Object.keys(attributesObj).forEach(key => {
                            if (key.startsWith(CONSTANTS.ATTRIBUTES.DATA) &&
                                (key.indexOf(CONSTANTS.ATTRIBUTES.DATA) !== -1 || key.indexOf(CONSTANTS.ATTRIBUTES.DATA2) !== -1)
                            ) {
                                cssSelector = getCssByAttribute(element, cssSelector, attributesObj, key);
                                stop = true;
                            }
                        });
                    }
                });

                if (stop) {
                    foundedElements = foundElementsByCssSelector(element, cssSelector, isElementInShadowDom);

                    if (foundedElements) {
                        cssSelector = getClarifiedCssSelector(cssSelector, element, foundedElements);

                        return cssSelector.trim();
                    }
                }
            }

            if (element.className) {
                let cssByClassNameInfo = getCssSelectorByClassName(element, isElementInShadowDom);

                if (cssByClassNameInfo) {
                    cssSelector = cssByClassNameInfo.cssSelector;
                    foundedElements = cssByClassNameInfo.foundedElements;

                    if (foundedElements) {
                        cssSelector = getClarifiedCssSelector(cssSelector, element, foundedElements);

                        return cssSelector.trim();
                    }
                }
            }

            cssSelector = nodeName;
            foundedElements = foundElementsByCssSelector(element, cssSelector, isElementInShadowDom);

            cssSelector = foundedElements ?
                getClarifiedCssSelector(cssSelector, element, foundedElements) : generateAbsoluteCssSelector(element);
        } catch (error) {
            cssSelector = generateAbsoluteCssSelector(element);
        }

        return cssSelector.trim();
    };

    const getCssSelectorByClassName = (element, isElementInShadowDom) => {
        let cssSelector = element.nodeName.toLowerCase();
        let classes = element.className.trim();

        let validClasses = getValidCssClasses(classes);
        if (validClasses) {
            cssSelector += '.'.concat(validClasses.replaceAll(' ', '.'));
            return {
                cssSelector,
                foundedElements: foundElementsByCssSelector(element, cssSelector, isElementInShadowDom)
            };
        }

        return null;
    };

    const foundElementsByCssSelector = (element, cssSelector, isElementInShadowDom) => {
        let result;

        try {
            result = isElementInShadowDom ?
                element.getRootNode().host.shadowRoot.querySelectorAll(cssSelector) :
                element.ownerDocument.querySelectorAll(cssSelector);
        } catch (error) {
            result = null;
        }

        return result;
    };

    const removeLineBreaks = text => {
        let arr = text.split('\n');
        return arr[0].length > 0 ? arr[0] : arr[1];
    };

    const getCssByAttribute = (element, initialCssSelector, attributesObj, attr) => {
        let name = attr;
        let nodeValue = attributesObj[attr];
        nodeValue = removeLineBreaks(nodeValue);

        return nodeValue.includes('\'') ?
            CONSTANTS.CSS_HELPERS.CSS_SELECTOR_1
                .replace(CONSTANTS.REPLACE_STRINGS.CSS, initialCssSelector)
                .replace(CONSTANTS.REPLACE_STRINGS.NAME, name)
                .replace(CONSTANTS.REPLACE_STRINGS.VALUE, nodeValue.trim()) :
            CONSTANTS.CSS_HELPERS.CSS_SELECTOR_2
                .replace(CONSTANTS.REPLACE_STRINGS.CSS, initialCssSelector)
                .replace(CONSTANTS.REPLACE_STRINGS.NAME, name)
                .replace(CONSTANTS.REPLACE_STRINGS.VALUE, nodeValue);
    };

    const generateAbsoluteCssSelector = element => {
        let result = '';
        let nodeName = getNodeName(element);
        let parentNodeName = getNodeName(element.parentNode);
        let children = element.parentNode.children;

        Array.from(children).forEach((child, index) => {
            if (child === element) {
                if (parentNodeName !== CONSTANTS.TAGS.HTML &&
                    parentNodeName !== CONSTANTS.TAGS.BODY &&
                    parentNodeName !== CONSTANTS.TAGS.DOCUMENT_FRAGMENT) {
                    result = CONSTANTS.CSS_HELPERS.NTH_CHILD_FULL
                        .replace(CONSTANTS.REPLACE_STRINGS.CSS, generateAbsoluteCssSelector(element.parentNode))
                        .replace(CONSTANTS.REPLACE_STRINGS.NAME, nodeName)
                        .replace(CONSTANTS.REPLACE_STRINGS.VALUE, (index + 1).toString());
                    result = result.trim().charAt(0) === '>' ?
                        result.trim().slice(1) :
                        result;
                    return true;
                }
            }
        });

        return result;
    };

    const getClarifiedCssSelector = (initialCssSelector, element, foundedElements) => {
        let result = initialCssSelector;
        if (foundedElements.length > 1) {
            let index = Array.from(foundedElements).indexOf(element);
            result += CONSTANTS.CSS_HELPERS.NTH_OF_TYPE
                .replace(CONSTANTS.REPLACE_STRINGS.VALUE, String(index + 1));
        }

        return result;
    };

    const getShadowHosts = element => {
        let result = [];
        let hostSelector;
        let isElementInShadowDom;
        let host = element.parentNode.getRootNode().host;

        while (host) {
            isElementInShadowDom = subject7Gutenberg.locatorRecorder.isInShadowRoot(host);
            hostSelector = generateCssSelector(host, isElementInShadowDom);
            result.push(hostSelector);
            host = host.parentNode ? host.parentNode.getRootNode().host : null;
        }

        result.reverse()

        return result;
    };

    const isValidIdentifier = (id, isClass) => {
        let regex = new RegExp(CONSTANTS.ID_RULES.CONTAINS_RESTRICTED_CHARACTERS, 'gi');

        if (regex.test(id)) {
            return false;
        }

        regex = new RegExp(CONSTANTS.ID_RULES.STARTS_WITH_NUMBER, 'gi');
        if (regex.test(id)) {
            return false;
        }

        regex = new RegExp(CONSTANTS.ID_RULES.ENDS_WITH_COLON_AND_NUMBER, 'gi');
        if (regex.test(id)) {
            return true;
        }

        regex = new RegExp(CONSTANTS.ID_RULES.ENDS_WITH_NUMBER, 'gi');
        if (regex.test(id) && !isClass) {
            return false;
        }

        regex = new RegExp(CONSTANTS.ID_RULES.STARTS_WITH_HYPHEN_AND_NUMBER, 'gi');

        return !regex.test(id);
    };

    const getValidCssClasses = classes => {
        let validClasses = '';
        let classesArr = classes.split(' ');

        classesArr.forEach(clazz => {
            if (isValidIdentifier(clazz, true)) {
                validClasses += ' '.concat(clazz);
            }
        });

        return validClasses.trim();
    }

    const getNodeName = el => {
        return el.nodeName.toLowerCase();
    };

    const getElementByCss = (css, targetDocument) => {
        let elements;
        try {
            elements = css.includes(CONSTANTS.CSS.SHADOW_PSEUDO) ?
                findElementInShadowDom(css, targetDocument) :
                targetDocument.querySelectorAll(css);
        } catch (e) {
            switch (e.message) {
                case CONSTANTS.ERROR_MESSAGES.SHADOW_CSS_CANNOT_END_ON_A_SHADOW_ROOT:
                    throw {code: 2, message: e.message};
                case CONSTANTS.ERROR_MESSAGES.ELEMENT_NOT_FOUND_CSS:
                    throw {code: 1, message: e.message};
                default:
                    throw {code: 2, message: CONSTANTS.ERROR_MESSAGES.INVALID_CSS.concat(e.message)};
            }
        }

        if (elements.length === 0) {
            throw {code: 1, message: CONSTANTS.ERROR_MESSAGES.ELEMENT_NOT_FOUND_CSS};
        } else {
            if (elements.length === 1) {
                return elements[0];
            }

            throw {code: 0, message: CONSTANTS.ERROR_MESSAGES.CSS_RETURNS_SEVERAL_ELEMENTS, elements: elements};
        }
    };

    const findElementInShadowDom = (css, targetDocument) => {
        let targetElement = targetDocument;
        let tokenizedCss = tokenizeSelector(css);
        let cssArray = getCssArrayFromCssSelector(tokenizedCss);

        if (cssArray[cssArray.length - 1].includes(CONSTANTS.CSS.SHADOW_PSEUDO)) {
            throw new Error(CONSTANTS.ERROR_MESSAGES.SHADOW_CSS_CANNOT_END_ON_A_SHADOW_ROOT);
        }

        targetElement = getTargetElement(targetElement, cssArray);

        if (!targetElement) {
            throw new Error(CONSTANTS.ERROR_MESSAGES.ELEMENT_NOT_FOUND_CSS);
        }

        return [targetElement];
    };

    const getCssArrayFromCssSelector = tokenizedCss => {
        let result = [];
        let cssElement = '';
        tokenizedCss.forEach((token, index) => {
            if (token.type === CONSTANTS.TOKENS.TYPE ||
                token.type === CONSTANTS.TOKENS.PSEUDO_ELEMENT ||
                token.type === CONSTANTS.TOKENS.PSEUDO_CLASS ||
                token.type === CONSTANTS.TOKENS.CLASS ||
                token.type === CONSTANTS.TOKENS.ATTRIBUTE ||
                token.type === CONSTANTS.TOKENS.ID ||
                (token.type === CONSTANTS.TOKENS.COMBINATOR && token.content === ' > ')) {
                cssElement += token.content;
            }
            if (token.type === CONSTANTS.TOKENS.COMBINATOR && token.content === ' ') {
                result.push(cssElement);
                cssElement = '';
            }
            if (index === tokenizedCss.length - 1) {
                result.push(cssElement);
                cssElement = '';
            }
        });

        return result;
    };

    const tokenizeSelector = selector => {
        selector = selector.trim();
        if (selector === '') {
            return [];
        }
        const replacements = [];
        // Replace escapes with placeholders
        selector = selector.replace(/\\./g, (value, offset) => {
            replacements.push({value, offset});
            return '\uE000'.repeat(value.length);
        });
        // Replace strings with placeholders
        selector = selector.replace(/(['"])([^\\\n]+?)\1/g, (value, quote, content, offset) => {
            replacements.push({value, offset});
            return `${quote}${'\uE001'.repeat(content.length)}${quote}`;
        });
        // Replace parentheses with placeholders
        {
            let pos = 0;
            let offset;
            while ((offset = selector.indexOf('(', pos)) > -1) {
                const value = gobbleParens(selector, offset);
                replacements.push({value, offset});
                selector = `${selector.substring(0, offset)}(${'¶'.repeat(value.length - 2)})${selector.substring(offset + value.length)}`;
                pos = offset + value.length;
            }
        }
        // Now we have no nested structures, and we can parse with regexes
        const tokens = tokenizeBy(selector);
        // Replace placeholders in reverse order
        const changedTokens = new Set();
        for (const replacement of replacements.reverse()) {
            for (const token of tokens) {
                const {offset, value} = replacement;
                if (!(token.pos[0] <= offset &&
                    offset + value.length <= token.pos[1])) {
                    continue;
                }
                const {content} = token;
                const tokenOffset = offset - token.pos[0];
                token.content =
                    content.slice(0, tokenOffset) +
                    value +
                    content.slice(tokenOffset + value.length);
                if (token.content !== content) {
                    changedTokens.add(token);
                }
            }
        }
        // Update changed tokens
        for (const token of changedTokens) {
            const pattern = getArgumentPatternByType(token.type);
            if (!pattern) {
                throw new Error(`Unknown token type: ${token.type}`);
            }
            pattern.lastIndex = 0;
            const match = pattern.exec(token.content);
            if (!match) {
                throw new Error(`Unable to parse content for ${token.type}: ${token.content}`);
            }
            Object.assign(token, match.groups);
        }

        tokens.forEach(token => {
            delete token.namespace;
        });

        return tokens;
    };

    const gobbleParens = (text, offset) => {
        let nesting = 0;
        let result = '';
        for (; offset < text.length; offset++) {
            const char = text[offset];
            switch (char) {
                case '(':
                    ++nesting;
                    break;
                case ')':
                    --nesting;
                    break;
            }
            result += char;
            if (nesting === 0) {
                return result;
            }
        }
        return result;
    };

    const tokenizeBy = text => {
        if (!text) {
            return [];
        }

        const tokens = [text];
        for (const [type, pattern] of Object.entries(CONSTANTS.TOKEN_SIGNATURES)) {
            for (let i = 0; i < tokens.length; i++) {
                const token = tokens[i];
                if (typeof token !== 'string') {
                    continue;
                }
                pattern.lastIndex = 0;
                const match = pattern.exec(token);
                if (!match) {
                    continue;
                }
                const from = match.index - 1;
                const args = [];
                const content = match[0];
                const before = token.slice(0, from + 1);
                if (before) {
                    args.push(before);
                }
                args.push({
                    ...match.groups,
                    type,
                    content,
                });
                const after = token.slice(from + content.length + 1);
                if (after) {
                    args.push(after);
                }
                tokens.splice(i, 1, ...args);
            }
        }
        let offset = 0;
        for (const token of tokens) {
            switch (typeof token) {
                case 'string':
                    throw new Error(`Unexpected sequence ${token} found at index ${offset}`);
                case 'object':
                    offset += token.content.length;
                    token.pos = [offset - token.content.length, offset];
                    break;
            }
        }
        return tokens;
    };

    const getArgumentPatternByType = type => {
        switch (type) {
            case 'pseudo-element':
            case 'pseudo-class':
                return new RegExp(CONSTANTS.TOKEN_SIGNATURES[type].source.replace('(?<argument>¶*)', '(?<argument>.*)'), 'gu');
            default:
                return CONSTANTS.TOKEN_SIGNATURES[type];
        }
    };

    const getTargetElement = (targetElement, cssArray) => {
        let index;
        let updatedItem;
        let isShadowHost;

        cssArray.forEach(item => {
            index = getIndexOfElementInSelector(item);
            updatedItem = getElementSelectorWithoutNthOfType(item);
            isShadowHost = updatedItem.includes(CONSTANTS.CSS.SHADOW_PSEUDO);

            targetElement = getTargetElementBySelector(targetElement, updatedItem, index, isShadowHost);
        });

        return targetElement;
    };

    const getTargetElementBySelector = (targetElement, selector, index, isShadowHost) => {
        if (isShadowHost) {
            selector = selector.replace(CONSTANTS.CSS.SHADOW_PSEUDO, '');
        }
        let target = index !== -1 ?
            targetElement.querySelectorAll(selector)[index] :
            targetElement.querySelector(selector);

        return isShadowHost ? target.shadowRoot : target;
    };

    const getIndexOfElementInSelector = (element, forSlot) => {
        const regex = new RegExp(CONSTANTS.CSS_HELPERS.NTH_PATTERN, 'gi');
        const resultArr = regex.exec(element);

        if (resultArr && resultArr[1]) {
            return resultArr[1].toString() - 1;
        }

        if (element.includes('#') || element.includes('.')) {
            return -1;
        }

        return forSlot ? -1 : 0;
    }

    const getElementSelectorWithoutNthOfType = element => {
        const regex = new RegExp(CONSTANTS.CSS_HELPERS.NTH_PATTERN, 'gi');
        const resultArr = regex.exec(element);
        if (resultArr && resultArr[0]) {
            return element.replace(resultArr[0], '');
        }

        return element;
    };

    const isValidXPath = expr => (
        typeof expr !== 'undefined' &&
        expr.replace(/[\s-_=]/g, '') !== '' &&
        expr.length === expr.replace(/[-_\w:.]+\(\)\s*=|=\s*[-_\w:.]+\(\)|\sor\s|\sand\s|\[(?:[^\/\]]+[\/\[]\/?.+)+\]|starts-with\(|\[.*last\(\)\s*[-\+<>=].+\]|number\(\)|not\(|count\(|text\(|first\(|normalize-space|[^\/]following-sibling|concat\(|descendant::|parent::|self::|child::|/gi, '').length
    );

    const getValidationRegexp = () => {
        let regex =
            "(?P<node>" +
            "(" +
            "^id\\([\"\\']?(?P<idvalue>%(value)s)[\"\\']?\\)" +// special case! `id(idValue)`
            "|" +
            "(?P<nav>//?(?:following-sibling::)?)(?P<tag>%(tag)s)" + //  `//div`
            "(\\[(" +
            "(?P<matched>(?P<mattr>@?%(attribute)s=[\"\\'](?P<mvalue>%(value)s))[\"\\']" + // `[@id="well"]` supported and `[text()="yes"]` is not
            "|" +
            "(?P<contained>contains\\((?P<cattr>@?%(attribute)s,\\s*[\"\\'](?P<cvalue>%(value)s)[\"\\']\\))" +// `[contains(@id, "bleh")]` supported and `[contains(text(), "some")]` is not
            ")\\])?" +
            "(\\[\\s*(?P<nth>\\d|last\\(\\s*\\))\\s*\\])?" +
            ")" +
            ")";

        const subRegexes = {
            "tag": "([a-zA-Z][a-zA-Z0-9]{0,10}|\\*)",
            "attribute": "[.a-zA-Z_:][-\\w:.]*(\\(\\))?)",
            "value": "\\s*[\\w/:][-/\\w\\s,:;.]*"
        };

        Object.keys(subRegexes).forEach(key => {
            regex = regex.replace(new RegExp('%\\(' + key + '\\)s', 'gi'), subRegexes[key]);
        });

        regex = regex.replace(/\?P<node>|\?P<idvalue>|\?P<nav>|\?P<tag>|\?P<matched>|\?P<mattr>|\?P<mvalue>|\?P<contained>|\?P<cattr>|\?P<cvalue>|\?P<nth>/gi, '');

        return new RegExp(regex, 'gi');
    };

    const preParseXpath = expr => (
        expr.replace(/contains\s*\(\s*concat\(["']\s+["']\s*,\s*@class\s*,\s*["']\s+["']\)\s*,\s*["']\s+([a-zA-Z0-9-_]+)\s+["']\)/gi, '@class="$1"')
    );

    const xPathToCss = async expr => {
        if (!expr) {
            throw new Error(CONSTANTS.ERROR_MESSAGES.MISSING_XPATH_EXPRESSION);
        }

        expr = preParseXpath(expr);

        if (!isValidXPath(expr)) {
            throw new Error(CONSTANTS.ERROR_MESSAGES.INVALID_OR_UNSUPPORTED_XPATH + expr);
        }

        const xPathArr = expr.split('|');
        const validationRegexp = getValidationRegexp();
        const cssSelectors = [];
        let xIndex = 0;

        while (xPathArr[xIndex]) {
            const css = [];
            let position = 0;
            let nodes;

            while (nodes = validationRegexp.exec(xPathArr[xIndex])) {
                let attr;

                if (!nodes && position === 0) {
                    throw new Error(CONSTANTS.ERROR_MESSAGES.INVALID_OR_UNSUPPORTED_XPATH + expr);
                }

                const match = {
                    node: nodes[5],
                    idvalue: nodes[12] || nodes[3],
                    nav: nodes[4],
                    tag: nodes[5],
                    matched: nodes[7],
                    mattr: nodes[10] || nodes[14],
                    mvalue: nodes[12] || nodes[16],
                    contained: nodes[13],
                    cattr: nodes[14],
                    cvalue: nodes[16],
                    nth: nodes[18]
                };

                let nav = '';

                if (position !== 0 && match['nav']) {
                    if (~match['nav'].indexOf('following-sibling::')) {
                        nav = ' + ';
                    } else {
                        nav = (match['nav'] === '//') ? ' ' : ' > ';
                    }
                }

                const tag = (match['tag'] === '*') ? '' : (match['tag'] || '');

                if (match['contained']) {
                    if (match['cattr'].indexOf('@') === 0) {
                        attr = '[' + match['cattr'].replace(/^@/, '') + '*="' + match['cvalue'] + '"]';
                    } else {
                        throw new Error(CONSTANTS.ERROR_MESSAGES.INVALID_OR_UNSUPPORTED_XPATH_ATTRIBUTE + match['cattr']);
                    }
                } else {
                    if (match['matched']) {
                        switch (match['mattr']) {
                            case '@id':
                                attr = '#' + match['mvalue'].replace(/^\s+|\s+$/, '').replace(/\s/g, '#');
                                break;
                            case '@class':
                                attr = '.' + match['mvalue'].replace(/^\s+|\s+$/, '').replace(/\s/g, '.');
                                break;
                            case 'text()':
                            case '.':
                                throw new Error(CONSTANTS.ERROR_MESSAGES.INVALID_OR_UNSUPPORTED_XPATH_ATTRIBUTE + match['mattr']);
                            default:
                                if (match['mattr'].indexOf('@') !== 0) {
                                    throw new Error(CONSTANTS.ERROR_MESSAGES.INVALID_OR_UNSUPPORTED_XPATH_ATTRIBUTE + match['mattr']);
                                }
                                if (match['mvalue'].indexOf(' ') !== -1) {
                                    match['mvalue'] = '\"' + match['mvalue'].replace(/^\s+|\s+$/, '') + '\"';
                                }
                                attr = '[' + match['mattr'].replace('@', '') + '="' + match['mvalue'] + '"]';
                                break;
                        }
                    } else {
                        if (match['idvalue']) {
                            attr = '#' + match['idvalue'].replace(/\s/, '#');
                        } else {
                            attr = '';
                        }
                    }
                }

                let nth = '';

                if (match['nth']) {
                    if (match['nth'].indexOf('last') === -1) {
                        if (isNaN(parseInt(match['nth'], 10))) {
                            throw new Error(CONSTANTS.ERROR_MESSAGES.INVALID_OR_UNSUPPORTED_XPATH_ATTRIBUTE + match['nth']);
                        }
                        nth = parseInt(match['nth'], 10) !== 1 ? ':nth-of-type(' + match['nth'] + ')' : ':first-of-type';
                    } else {
                        nth = ':last-of-type';
                    }
                }

                css.push(nav + tag + attr + nth);
                position++;
            }

            const result = css.join('');

            if (result === '') {
                throw new Error(CONSTANTS.ERROR_MESSAGES.INVALID_OR_UNSUPPORTED_XPATH);
            }

            cssSelectors.push(result);
            xIndex++;
        }

        let cssSelector = cssSelectors.join(', ');
        let element = await subject7Gutenberg.locatorRecorder.testCssSelectorAfterGeneration(cssSelector);

        return element ? cssSelector : null;
    };

    return {
        generateCssForElementInShadowDom: generateCssForElementInShadowDom,
        getElementByCss: getElementByCss,
        getNodeName: getNodeName,
        isValidIdentifier: isValidIdentifier,
        xPathToCss: xPathToCss
    };
})();

subject7Gutenberg.locatorRecorder = (() => {
    let target = null;
    let originEvent = null;
    let targetWindow = null;
    let shadowDomMode = CONSTANTS.SHADOW_MODES.OPEN;
    let isElementInShadowDom = false;
    let targetComposedPath = null;
    let iframeDialogHint;
    let dialogHintContainer;
    const dialogHintBlockId = 'subject7dlgH';
    let iframeDialogLocator;
    let dialogLocatorContainer;
    const dialogLocatorBlockId = 'subject7dlgL';
    let lastTimeoutHandle = 0;
    let waitDblClickTimeoutId = 0;
    const successMessageBlockId = 'subject7GutenbergSuccessBlock';
    let successMessageBlock;
    const errorMessageBlockId = 'subject7GutenbergErrorBlock';
    let errorMessageBlock;

    const generateXpath = async hint => {
        if (targetWindow !== window) {
            return null;
        }

        const resultData = {};
        try {
            let result = null;
            if (!isElementInShadowDom) {
                result = await subject7Gutenberg.xpathEngine.generateXPath(target, hint.text, hint);
                resultData.xpath = result.xpath;
                resultData.hint = result.hint;
                resultData.logInfo = result.logInfo;
                resultData.errors = result.errors;
                resultData.patternId = result.patternId;
                resultData.patternDate = result.patternDate;
            } else {
                resultData.xpath = null;
                resultData.hint = null;
                resultData.logInfo = [];
                resultData.errors = [];
                resultData.patternId = null;
                resultData.patternDate = null;
            }
        } catch (error) {
            let isFrameset = await getIsFramesetProperty(window);
            if (isFrameset) {
                showMessage(error.message);
            } else {
                showErrorNotification(error.message);
            }
            return {hasError: true, message: error.message};
        }

        const frameXpathIfExist = await getIframeXpath(targetWindow);
        if (frameXpathIfExist !== null) {
            resultData.frame = frameXpathIfExist;
        }

        return resultData;
    };

    const getIframeXpath = async window => {
        const checkIsParentFrame = (paramWindow) => new Promise((resolve, reject) => {
            const channel = new MessageChannel();

            channel.port1.onmessage = event => {
                channel.port1.close();
                if (event.data.error) {
                    reject(event.data.error);
                } else {
                    resolve(event.data.result);
                }
            };

            paramWindow.parent.postMessage(
                {
                    sender: CONSTANTS.MESSAGE_SENDERS.CHILD_WINDOW,
                    action: CONSTANTS.ACTIONS.CHECK_IS_FRAME
                },
                '*',
                [channel.port2]
            );
        });

        let resultFrameAnalysis = getFrameInfo();
        let isParentFrame;
        if ((resultFrameAnalysis.crossOrigin && resultFrameAnalysis.sandboxed) ||
            (resultFrameAnalysis.crossOrigin && !resultFrameAnalysis.sandboxed) ||
            (!resultFrameAnalysis.crossOrigin && resultFrameAnalysis.sandboxed)) {
            if (resultFrameAnalysis.sandboxed && resultFrameAnalysis.sandboxAllowances &&
                (resultFrameAnalysis.sandboxAllowances.sameOrigin || resultFrameAnalysis.sandboxAllowances.scripts)) {
                isParentFrame = window.parent.location !== window.top.location;
            } else {
                isParentFrame = await checkIsParentFrame(window);
            }
        } else {
            isParentFrame = window.parent.location !== window.top.location;
        }

        if (isParentFrame) {
            let xpath = await getIframeXpath(window.parent);
            return xpath + ';' + await getFrameXpathIfExist(window);
        } else {
            return await getFrameXpathIfExist(window);
        }
    };

    const startCapturing = () => {
        target = null;
        originEvent = null;
        targetWindow = null;
        shadowDomMode = CONSTANTS.SHADOW_MODES.OPEN;
        isElementInShadowDom = false;
        targetComposedPath = null;
        blockUI(false);
        captureEvents(true);
        addDialogBlock(CONSTANTS.DIALOG_FRAME_OPTIONS.HINT.TYPE);
        addDialogBlock(CONSTANTS.DIALOG_FRAME_OPTIONS.LOCATOR.TYPE);
        addSuccessMessageBlock();
        addErrorMessageBlock();
        started = true;
    };

    const stopCapturing = () => {
        target = null;
        originEvent = null;
        targetWindow = null;
        shadowDomMode = CONSTANTS.SHADOW_MODES.OPEN;
        isElementInShadowDom = false;
        targetComposedPath = null;
        blockUI(false);
        captureEvents(false);
        removeDialogBlock(CONSTANTS.DIALOG_FRAME_OPTIONS.HINT.TYPE);
        removeDialogBlock(CONSTANTS.DIALOG_FRAME_OPTIONS.LOCATOR.TYPE);
        removeSuccessMessageBlock();
        removeErrorMessageBlock();
        started = false;
    };

    const resetCapturing = fireEvent => {
        if (targetWindow !== window) {
            return;
        }

        blockUI(false);
        if (fireEvent) {
            captureEvents(false);
            if (isElementInShadowDom) {
                targetComposedPath[0].dispatchEvent(originEvent);
            } else {
                target.dispatchEvent(originEvent);
            }
        }
        target = null;
        originEvent = null;
        targetWindow = null;
        shadowDomMode = CONSTANTS.SHADOW_MODES.OPEN;
        isElementInShadowDom = false;
        targetComposedPath = null;
        captureEvents(started);
    };

    const test = async (data, type, callback) => {
        if (targetWindow !== window) {
            return;
        }

        let element;
        try {
            element = type === CONSTANTS.TEST_TYPES.XPATH ?
                subject7Gutenberg.xpathEngine.getElementByXpath(data, targetWindow.document) :
                subject7Gutenberg.cssSelectorEngine.getElementByCss(data, targetWindow.document);
        } catch (error) {
            let isFrameset = await getIsFramesetProperty(window);
            if (isFrameset) {
                showMessage(error.message);
            } else {
                showErrorNotification(error.message);
            }
            callback();
            return;
        }
        scrollToElement(element);
        highlight(element, callback);
    };

    const testCssSelectorAfterGeneration = async data => {
        if (targetWindow !== window) {
            return null;
        }

        let element = null;
        try {
            element = subject7Gutenberg.cssSelectorEngine.getElementByCss(data, targetWindow.document);
        } catch (error) {
            throw new Error(CONSTANTS.ERROR_MESSAGES.CSS_SELECTOR_GENERATION_ERROR.concat(error.message));
        }

        return element;
    };

    const blockUI = enabled => {
        const blockUiId = 'subject7GutenbergBlockUI';
        if (window !== window.top) {
            window.top.postMessage({
                sender: CONSTANTS.MESSAGE_SENDERS.CHILD_WINDOW,
                action: CONSTANTS.ACTIONS.BLOCK_UI,
                enabled
            }, '*');
            return;
        }

        const documentToBLock = window.frameElement ? window.top.document : window.document;
        let blockElement;
        if (enabled) {
            if (blockElement) {
                return;
            }
            blockElement = documentToBLock.createElement(CONSTANTS.TAGS.RECORDER_BLOCK);
            blockElement.setAttribute(CONSTANTS.ATTRIBUTES.ID, blockUiId);
            blockElement.setAttribute(CONSTANTS.ATTRIBUTES.STYLE,
                'position: fixed; ' +
                'left: 0px; ' +
                'top: 0px; ' +
                'width: 100%; ' +
                'height: 100%; ' +
                'background: rgba(0, 0, 0, 0.5); ' +
                'display: flex; ' +
                'justify-content: center; ' +
                'align-items: center;' +
                'z-index: 2147483646; '
            );
            documentToBLock.body.appendChild(blockElement);
            documentToBLock.addEventListener(CONSTANTS.EVENTS.DOM.KEY_DOWN, handleBlockedKeyPress, true);
        } else {
            blockElement = documentToBLock.getElementById(blockUiId);
            if (blockElement) {
                documentToBLock.body.removeChild(blockElement);
            }
            documentToBLock.removeEventListener(CONSTANTS.EVENTS.DOM.KEY_DOWN, handleBlockedKeyPress, true);
        }
    };

    const handleBlockedKeyPress = event => {
        event.preventDefault();
    };

    const captureEvents = enabled => {
        if (enabled) {
            window.addEventListener(CONSTANTS.EVENTS.DOM.CLICK, handleEvent, true);
            window.addEventListener(CONSTANTS.EVENTS.WINDOW.MESSAGE, handleIframeEvents, true);
        } else {
            window.removeEventListener(CONSTANTS.EVENTS.DOM.CLICK, handleEvent, true);
            window.removeEventListener(CONSTANTS.EVENTS.WINDOW.MESSAGE, handleIframeEvents, true);
        }
    };

    const getFrameInfo = () => {
        return sandblaster.detect() || {};
    };

    const handleIframeEvents = event => {
        const {data} = event;
        const {sender} = data;

        if (sender !== CONSTANTS.MESSAGE_SENDERS.CHILD_WINDOW) {
            return;
        }
        const {action} = data;
        switch (action) {
            case CONSTANTS.ACTIONS.BLOCK_UI:
                blockUI(data.enabled);
                break;
            case CONSTANTS.ACTIONS.GET_PRECISE_XPATH_FOR_FRAME:
                let iframe = getIFrame(getAllFrames(), data.iframeName, data.iframeSrc);
                let clarifiedXpath = null;
                if (iframe !== null) {
                    const idAttrValue = iframe.getAttribute(CONSTANTS.ATTRIBUTES.ID);
                    const nameAttrValue = iframe.getAttribute(CONSTANTS.ATTRIBUTES.NAME);
                    const titleAttrValue = iframe.getAttribute(CONSTANTS.ATTRIBUTES.TITLE);
                    let iframeXpath = CONSTANTS.XPATH_HELPERS.START_XPATH.concat(iframe.tagName).toLowerCase();

                    if (idAttrValue && idAttrValue.trim().length && subject7Gutenberg.cssSelectorEngine.isValidIdentifier(idAttrValue.trim())) {
                        clarifiedXpath = `${iframeXpath}[@id='${idAttrValue}']`;
                    } else if (nameAttrValue && nameAttrValue.trim().length && subject7Gutenberg.cssSelectorEngine.isValidIdentifier(nameAttrValue.trim())) {
                        clarifiedXpath = `${iframeXpath}[@name='${nameAttrValue}']`;
                    } else if (titleAttrValue && titleAttrValue.trim().length) {
                        clarifiedXpath = `${iframeXpath}[@title='${titleAttrValue}']`;
                    } else {
                        let frames = iframe.ownerDocument.getElementsByTagName(CONSTANTS.TAGS.IFRAME);
                        let frameIndex = Array.from(frames).indexOf(iframe);
                        if (frameIndex !== -1) {
                            clarifiedXpath = `(${iframeXpath})[${frameIndex + 1}]`;
                        }
                    }
                }
                event.ports[0].postMessage({result: clarifiedXpath});
                event.ports[0].close();
                break;
            case CONSTANTS.ACTIONS.CHECK_IS_FRAME:
                let resultFrameAnalysis = getFrameInfo();
                event.ports[0].postMessage({result: resultFrameAnalysis.framed});
                event.ports[0].close();
                break;
            case CONSTANTS.ACTIONS.CHECK_IS_FRAMESET:
                event.ports[0].postMessage({result: window.document.body.nodeName === CONSTANTS.TAGS.FRAMESET});
                event.ports[0].close();
                break;
            default:
                break;
        }
    };

    const getFrameXpathIfExist = async paramWindow => {
        let frame;
        let iframeXpath;
        let resultFrameAnalysis = getFrameInfo();

        if (!resultFrameAnalysis.framed) {
            return null;
        }

        if (resultFrameAnalysis.framed && !resultFrameAnalysis.crossOrigin) {
            try {
                frame = paramWindow.frameElement;
            } catch (e) {
                frame = null;
            }

            if (frame !== null) {
                const sandboxAttrValue = frame.getAttribute(CONSTANTS.ATTRIBUTES.SANDBOX);
                const idAttrValue = frame.getAttribute(CONSTANTS.ATTRIBUTES.ID);
                const nameAttrValue = frame.getAttribute(CONSTANTS.ATTRIBUTES.NAME);
                const srcAttrValue = frame.getAttribute(CONSTANTS.ATTRIBUTES.SRC);
                const titleAttrValue = frame.getAttribute(CONSTANTS.ATTRIBUTES.TITLE);

                iframeXpath = CONSTANTS.XPATH_HELPERS.START_XPATH.concat(frame.tagName).toLowerCase();

                if (idAttrValue && idAttrValue.trim().length && subject7Gutenberg.cssSelectorEngine.isValidIdentifier(idAttrValue.trim())) {
                    return `${iframeXpath}[@id='${idAttrValue}']`;
                }
                if (nameAttrValue && nameAttrValue.trim().length && subject7Gutenberg.cssSelectorEngine.isValidIdentifier(nameAttrValue.trim())) {
                    return `${iframeXpath}[@name='${nameAttrValue}']`;
                }
                if (titleAttrValue && titleAttrValue.trim().length) {
                    return `${iframeXpath}[@title='${titleAttrValue}']`;
                }
                if (resultFrameAnalysis.sandboxed && sandboxAttrValue) {
                    return sandboxAttrValue.trim().length ?
                        `${iframeXpath}[@sandbox='${sandboxAttrValue}']` :
                        `${iframeXpath}[@sandbox]`;
                }

                let frames = frame.ownerDocument.getElementsByTagName(CONSTANTS.TAGS.IFRAME);
                let frameIndex = Array.from(frames).indexOf(frame);
                if (frameIndex !== -1) {
                    return `(${iframeXpath})[${frameIndex + 1}]`;
                }

                if (srcAttrValue && srcAttrValue.trim().length && srcAttrValue !== CONSTANTS.ATTRIBUTE_SRC_VALUE_SRCDOC) {
                    return `${iframeXpath}[@src='${srcAttrValue}']`;
                }
            }

            return await getPreciseXpathForFrame(paramWindow, resultFrameAnalysis);
        }

        return await getPreciseXpathForFrame(paramWindow, resultFrameAnalysis);
    };

    const getPreciseXpathForFrame = async (paramWindow, resultFrameAnalysis) => {
        let iframeXpath = CONSTANTS.XPATH_HELPERS.START_XPATH + CONSTANTS.TAGS.IFRAME;
        try {
            const getPreciseXpath = (iframeName, iframeSrc) => new Promise((resolve, reject) => {
                const channel = new MessageChannel();

                channel.port1.onmessage = event => {
                    channel.port1.close();
                    if (event.data.error) {
                        reject(event.data.error);
                    } else {
                        resolve(event.data.result);
                    }
                };

                paramWindow.parent.postMessage(
                    {
                        sender: CONSTANTS.MESSAGE_SENDERS.CHILD_WINDOW,
                        action: CONSTANTS.ACTIONS.GET_PRECISE_XPATH_FOR_FRAME,
                        iframeSrc,
                        iframeName,
                    },
                    '*',
                    [channel.port2]
                );
            });

            let iframeSrc = paramWindow.location.href;
            let preciseIframeXpath = await getPreciseXpath(paramWindow.name, iframeSrc);
            if (preciseIframeXpath) {
                return preciseIframeXpath;
            }
            if (iframeSrc !== CONSTANTS.ATTRIBUTE_SRC_VALUE_SRCDOC) {
                return `${iframeXpath}[@src='${iframeSrc}']`
            }

            throw `iframe src attribute has value "${CONSTANTS.ATTRIBUTE_SRC_VALUE_SRCDOC}"`;
        } catch (e) {
            return resultFrameAnalysis.sandboxed ?
                getIframeXpathBySandboxAllowances(iframeXpath, resultFrameAnalysis.sandboxAllowances) :
                null;
        }
    };

    const getIframeXpathBySandboxAllowances = (iframeXpath, sandboxAllowances) => {
        if (sandboxAllowances) {
            if (sandboxAllowances.sameOrigin) {
                return `${iframeXpath}[contains(@sandbox, 'allow-same-origin')]`;
            }
            if (sandboxAllowances.forms) {
                return `${iframeXpath}[contains(@sandbox, 'allow-forms')]`;
            }
            if (sandboxAllowances.modals) {
                return `${iframeXpath}[contains(@sandbox, 'allow-modals')]`;
            }
            if (sandboxAllowances.orientationLock) {
                return `${iframeXpath}[contains(@sandbox, 'allow-orientation-lock')]`;
            }
            if (sandboxAllowances.pointerLock) {
                return `${iframeXpath}[contains(@sandbox, 'allow-pointer-lock')]`;
            }
            if (sandboxAllowances.popups) {
                return `${iframeXpath}[contains(@sandbox, 'allow-popups')]`;
            }
            if (sandboxAllowances.popupsToEscapeSandbox) {
                return `${iframeXpath}[contains(@sandbox, 'allow-popups-to-escape-sandbox')]`;
            }
            if (sandboxAllowances.presentation) {
                return `${iframeXpath}[contains(@sandbox, 'allow-presentation')]`;
            }
            if (sandboxAllowances.topNavigation) {
                return `${iframeXpath}[contains(@sandbox, 'allow-top-navigation')]`;
            }
            if (sandboxAllowances.topNavigationByUserActivation) {
                return `${iframeXpath}[contains(@sandbox, 'allow-top-navigation-by-user-activation')]`;
            }
        }

        return `${iframeXpath}[@sandbox]`;
    };

    const getAllFrames = () => {
        return document.getElementsByTagName(CONSTANTS.TAGS.IFRAME);
    };

    const getIFrame = (iframes, iframeName, iframeSrc) => {
        for (const f of iframes) {
            if ((iframeName.length !== 0 && f.name === iframeName) || iframeSrc === f.getAttribute(CONSTANTS.ATTRIBUTES.SRC)) {
                return f;
            }
        }

        return null;
    };

    const handleEvent = event => {
        if (checkIfTargetElementIsDialogFrame(event.target.id)) {
            return;
        }

        if (checkIfTargetElementIsHtmlOrBody(event.target)) {
            return;
        }

        defineShadowRootVariables(event.composedPath());
        target = findClickableElement(event.target);

        if (checkIfTargetElementIsHtmlOrBody(target)) {
            return;
        }

        originEvent = new event.constructor(event.type, event);
        targetWindow = window;
        event.preventDefault();
        event.stopPropagation();

        clearTimeout(waitDblClickTimeoutId);
        waitDblClickTimeoutId = setTimeout(async () => {
            captureEvents(false);
            blockUI(true);
            const isFrameset = await getIsFramesetProperty(window);
            chrome.runtime.sendMessage({
                action: CONSTANTS.ACTIONS.ELEMENT_CAPTURED,
                isFrameset,
                isDisabledHint: isElementInShadowDom || shadowDomMode.includes(CONSTANTS.SHADOW_MODES.CLOSED),
                isDisabledGenerating: shadowDomMode.includes(CONSTANTS.SHADOW_MODES.CLOSED)
            });
        }, 320);
    };

    const isInShadowRoot = node => {
        let parent = node.parentNode;

        while (parent) {
            if (parent.toString() === '[object ShadowRoot]') {
                shadowDomMode = parent.mode.toString();
                return true;
            }
            parent = parent.parentNode;
        }

        return false;
    };

    const isSVGElement = element => {
        return element instanceof SVGElement;
    };
    const containsCustomNamespace = element => {
        const hasNamespaceURI = element.namespaceURI === CONSTANTS.NAMESPACE_URI_DEFAULT;
        const regex = /^(?!:)[^:]+:[^:]+(?<!:)$/;
        const hasColon = regex.test(getTagName(element));
        return !hasNamespaceURI || (hasNamespaceURI && hasColon);
    };

    const getOpenOrClosedShadowRoot = element => {
        return isSVGElement(element) || containsCustomNamespace(element) ?
            null :
            chrome.dom.openOrClosedShadowRoot(element);
    };

    const defineShadowRootVariables = composedPath => {
        targetComposedPath = composedPath;
        targetComposedPath.pop();
        targetComposedPath = targetComposedPath
            .filter(element => element.nodeType !== Node.DOCUMENT_NODE)
            .filter(element => subject7Gutenberg.cssSelectorEngine.getNodeName(element) !== CONSTANTS.TAGS.HTML)
            .filter(element => subject7Gutenberg.cssSelectorEngine.getNodeName(element) !== CONSTANTS.TAGS.BODY);

        shadowDomMode = getOpenOrClosedShadowRoot(targetComposedPath[0]) === null ?
            CONSTANTS.SHADOW_MODES.OPEN : CONSTANTS.SHADOW_MODES.CLOSED;
        isElementInShadowDom = shadowDomMode === CONSTANTS.SHADOW_MODES.OPEN ?
            isInShadowRoot(targetComposedPath[0]) : false;
    }

    const findClickableElement = element => {
        if (isClickableElement(element)) {
            return element;
        }

        let parent = element;

        while (parent.parentNode !== null) {
            if (isClickableElement(parent)) {
                return parent;
            }
            parent = parent.parentNode;
        }
        return element;
    };

    const isClickableElement = element => {
        return CONSTANTS.CLICKABLE_ELEMENT_TAGS.includes(element.localName.toLowerCase()) ||
            element.onclick !== null ||
            window.getComputedStyle(element).cursor === 'pointer';
    };

    const highlight = (element, callback) => {
        let count = 0;
        let originOutline = element.getAttribute(CONSTANTS.ATTRIBUTES.OUTLINE);
        let timer = setInterval(() => {
            if (count === 8) {
                clearInterval(timer);
                element.style.setProperty(CONSTANTS.ATTRIBUTES.OUTLINE, originOutline ? originOutline : '');
                callback();
                return;
            }
            if (count % 2 === 0) {
                element.style.setProperty(CONSTANTS.ATTRIBUTES.OUTLINE, '3px solid rgb(86, 135, 168)', 'important');
            } else {
                element.style.setProperty(CONSTANTS.ATTRIBUTES.OUTLINE, originOutline ? originOutline : '');
            }
            count++;
        }, 100);
    };

    const scrollToElement = element => {
        const elementRect = element.getBoundingClientRect();
        const absoluteElementTop = elementRect.top + window.pageYOffset;
        const middle = absoluteElementTop - (window.innerHeight / 2);
        window.scrollTo(0, middle);
    };

    const addDialogBlock = dialogType => {
        if (window.location !== window.parent.location) {
            return;
        }

        const documentToBLock = window.frameElement ? window.top.document : window.document;

        if (dialogType === CONSTANTS.DIALOG_FRAME_OPTIONS.HINT.TYPE &&
            documentToBLock.getElementById(dialogHintBlockId)) {
            return;
        }

        if (dialogType === CONSTANTS.DIALOG_FRAME_OPTIONS.LOCATOR.TYPE &&
            documentToBLock.getElementById(dialogLocatorBlockId)) {
            return;
        }

        let dialogBlockId = dialogType === CONSTANTS.DIALOG_FRAME_OPTIONS.HINT.TYPE ?
            dialogHintBlockId : dialogLocatorBlockId;

        let dialogContainer = documentToBLock.createElement(CONSTANTS.TAGS.RECORDER_BLOCK);
        dialogContainer.setAttribute(CONSTANTS.ATTRIBUTES.ID, dialogBlockId);
        dialogContainer.setAttribute(CONSTANTS.ATTRIBUTES.CLASS,
            'draggable resizable '.concat(dialogType === CONSTANTS.DIALOG_FRAME_OPTIONS.HINT.TYPE ?
                CONSTANTS.DIALOG_FRAME_OPTIONS.HINT.CLASS_NAME : CONSTANTS.DIALOG_FRAME_OPTIONS.LOCATOR.CLASS_NAME)
        );
        let width = dialogType === CONSTANTS.DIALOG_FRAME_OPTIONS.HINT.TYPE ?
            CONSTANTS.DIALOG_FRAME_OPTIONS.HINT.WIDTH : CONSTANTS.DIALOG_FRAME_OPTIONS.LOCATOR.WIDTH;
        dialogContainer.setAttribute(CONSTANTS.ATTRIBUTES.STYLE,
            'box-sizing: border-box; ' +
            'overflow: hidden; ' +
            'width: ' + width + 'px; ' +
            'visibility: hidden; ' +
            'z-index: 2147483647;' +
            'border-radius: 5px; ' +
            'background: white; ' +
            'display: block; ' +
            'left: 0;' +
            'top: 0; '
        );

        let handle = documentToBLock.createElement(CONSTANTS.TAGS.DIV);
        handle.setAttribute(CONSTANTS.ATTRIBUTES.ID, dialogBlockId.concat(CONSTANTS.DIALOG_ELEMENTS.HANDLE.LABEL));
        handle.setAttribute(CONSTANTS.ATTRIBUTES.CLASS,
            'handle '.concat(dialogType === CONSTANTS.DIALOG_FRAME_OPTIONS.HINT.TYPE ?
                CONSTANTS.DIALOG_FRAME_OPTIONS.HINT.CLASS_NAME : CONSTANTS.DIALOG_FRAME_OPTIONS.LOCATOR.CLASS_NAME)
        );
        handle.setAttribute(CONSTANTS.ATTRIBUTES.STYLE,
            'box-sizing: border-box; ' +
            'width: 100%; ' +
            'border: none; ' +
            'height: 40px; ' +
            'position: absolute; ' +
            'top: 0; ' +
            'left: 0; ' +
            'padding: 8px 15px 8px; ' +
            'display: flex; ' +
            'flex-direction: row; ' +
            'justify-content: space-between; ' +
            'align-items: center; ' +
            'background-color: #3975b5; ' +
            'cursor: move; ' +
            'z-index: 1; '
        );

        let headerText = documentToBLock.createElement(CONSTANTS.TAGS.LABEL);
        headerText.setAttribute(CONSTANTS.ATTRIBUTES.STYLE,
            'font-size: 18px; ' +
            'font-weight: 400; ' +
            'font-style: normal; ' +
            'font-family: "Open Sans", "Helvetica Neue", Helvetica, Arial, sans-serif; ' +
            'color: white; ' +
            'pointer-events: none;'
        );
        headerText.innerHTML = dialogType;
        handle.append(headerText);

        let closeContainer = documentToBLock.createElement(CONSTANTS.TAGS.DIV);
        closeContainer.setAttribute(CONSTANTS.ATTRIBUTES.ID, dialogBlockId.concat(CONSTANTS.DIALOG_ELEMENTS.CLOSE_CONTAINER.LABEL));
        closeContainer.setAttribute(CONSTANTS.ATTRIBUTES.STYLE,
            'width: 24px; ' +
            'height: 24px; ' +
            'background-color: #3975b5; ' +
            'display: flex; ' +
            'flex-direction: row; ' +
            'justify-content: center; ' +
            'align-items: center; ' +
            'cursor: default;'
        );
        closeContainer.setAttribute(CONSTANTS.ATTRIBUTES.ONMOUSEENTER, 'this.style.backgroundColor = \'#4984c3\';');
        closeContainer.setAttribute(CONSTANTS.ATTRIBUTES.ONMOUSELEAVE, 'this.style.backgroundColor = \'#3975b5\';');
        closeContainer.addEventListener(CONSTANTS.EVENTS.DOM.CLICK, () => {
            if (dialogType === CONSTANTS.DIALOG_FRAME_OPTIONS.HINT.TYPE) {
                chrome.runtime.sendMessage({
                    action: CONSTANTS.ACTIONS.HINT_DIALOG_RESULT,
                    skip: false,
                    isFrameset: false,
                    isDisabledHint: false
                });
            } else {
                chrome.runtime.sendMessage({
                    action: CONSTANTS.ACTIONS.LOCATOR_DIALOG_RESULT,
                    result: {
                        type: CONSTANTS.ACTIONS.SKIP,
                        skip: false,
                        isFrameset: false,
                    }
                });
            }
        });

        let closeBtn = documentToBLock.createElement(CONSTANTS.TAGS.LABEL);
        closeBtn.setAttribute(CONSTANTS.ATTRIBUTES.ID, dialogBlockId.concat(CONSTANTS.DIALOG_ELEMENTS.CLOSE_BUTTON.LABEL));
        closeBtn.setAttribute(CONSTANTS.ATTRIBUTES.STYLE,
            'font-size: 14px; ' +
            'font-weight: 600; ' +
            'font-style: normal; ' +
            'font-family: "Open Sans", "Helvetica Neue", Helvetica, Arial, sans-serif; ' +
            'color: white;'
        );
        closeBtn.innerHTML = '&#x2715;';
        closeContainer.append(closeBtn);

        handle.append(closeContainer);

        dialogContainer.append(handle);

        let iframeDialog = documentToBLock.createElement(CONSTANTS.TAGS.IFRAME);
        iframeDialog.setAttribute(CONSTANTS.ATTRIBUTES.ID, dialogBlockId.concat(CONSTANTS.DIALOG_ELEMENTS.IFRAME.LABEL));
        iframeDialog.setAttribute(CONSTANTS.ATTRIBUTES.SRC,
            chrome.runtime.getURL(dialogType === CONSTANTS.DIALOG_FRAME_OPTIONS.HINT.TYPE ?
                CONSTANTS.DIALOG_FRAME_OPTIONS.HINT.URL :
                CONSTANTS.DIALOG_FRAME_OPTIONS.LOCATOR.URL
            )
        );
        iframeDialog.setAttribute(CONSTANTS.ATTRIBUTES.STYLE,
            'box-sizing: border-box; ' +
            'height: calc(100% - 40px); ' +
            'width: ' + width + 'px; ' +
            'position: relative; ' +
            'top: auto; ' +
            'left: auto; ' +
            'margin-top: 40px; ' +
            'vertical-align: middle;' +
            'border: none; '
        );
        dialogContainer.append(iframeDialog);

        documentToBLock.body.append(dialogContainer);

        dialogContainer.style.position = 'fixed';

        let draggableOptions = Object.assign({}, CONSTANTS.DRAGGABLE_OPTIONS);
        switch (dialogType) {
            case CONSTANTS.DIALOG_FRAME_OPTIONS.HINT.TYPE:
                dialogHintContainer = dialogContainer;
                iframeDialogHint = iframeDialog;
                draggableOptions.handle = draggableOptions.handle.concat('.').concat(CONSTANTS.DIALOG_FRAME_OPTIONS.HINT.CLASS_NAME);
                $('.draggable.resizable.s7-hint').draggable(draggableOptions);
                break;
            case CONSTANTS.DIALOG_FRAME_OPTIONS.LOCATOR.TYPE:
                dialogLocatorContainer = dialogContainer;
                iframeDialogLocator = iframeDialog;
                draggableOptions.handle = draggableOptions.handle.concat('.').concat(CONSTANTS.DIALOG_FRAME_OPTIONS.LOCATOR.CLASS_NAME);
                $('.draggable.resizable.s7-locator').draggable(draggableOptions);
                break;
            default:
                break;
        }
    };

    const removeDialogBlock = dialogType => {
        if (dialogType === CONSTANTS.DIALOG_FRAME_OPTIONS.HINT.TYPE && !dialogHintContainer) {
            return;
        }

        if (dialogType === CONSTANTS.DIALOG_FRAME_OPTIONS.LOCATOR.TYPE && !dialogLocatorContainer) {
            return;
        }

        switch (dialogType) {
            case CONSTANTS.DIALOG_FRAME_OPTIONS.HINT.TYPE:
                dialogHintContainer.parentElement.removeChild(dialogHintContainer);
                dialogHintContainer = undefined;
                break;
            case CONSTANTS.DIALOG_FRAME_OPTIONS.LOCATOR.TYPE:
                dialogLocatorContainer.parentElement.removeChild(dialogLocatorContainer);
                dialogLocatorContainer = undefined;
                break;
            default:
                break;
        }
    };

    const openDialog = (dialogType, options) => {
        const {hint, locator, saveFeatureEnabled, isFrameset, isDisabledHint, isDisabledGenerating, algorithmType} = options;

        if (dialogType === CONSTANTS.DIALOG_FRAME_OPTIONS.HINT.TYPE && !dialogHintContainer) {
            return;
        }
        if (dialogType === CONSTANTS.DIALOG_FRAME_OPTIONS.LOCATOR.TYPE && !dialogLocatorContainer) {
            return;
        }

        const iframeDialog = dialogType === CONSTANTS.DIALOG_FRAME_OPTIONS.HINT.TYPE ?
            iframeDialogHint :
            iframeDialogLocator;

        clearTimeout(lastTimeoutHandle);
        closeDialog(dialogType);

        lastTimeoutHandle = setTimeout(
            () => {
                let message = dialogType === CONSTANTS.DIALOG_FRAME_OPTIONS.HINT.TYPE ?
                    {
                        sender: CONSTANTS.MESSAGE_SENDERS.S7_OPEN_HINT_WINDOW,
                        hint,
                        isFrameset,
                        isDisabledHint,
                        isDisabledGenerating,
                        saveFeatureEnabled
                    } :
                    {
                        sender: CONSTANTS.MESSAGE_SENDERS.S7_OPEN_LOCATOR_WINDOW,
                        hint,
                        locator,
                        saveFeatureEnabled,
                        isFrameset,
                        isDisabledHint,
                        algorithmType
                    };
                iframeDialog.contentWindow.postMessage(message, '*');
            },
            dialogType === CONSTANTS.DIALOG_FRAME_OPTIONS.HINT.TYPE ?
                CONSTANTS.OPEN_DIALOG_DELAY.HINT :
                CONSTANTS.OPEN_DIALOG_DELAY.LOCATOR
        );
    };

    const syncDialogSize = ({dialogType, dialogWidth, dialogHeight}) => {
        if (dialogType === CONSTANTS.DIALOG_FRAME_OPTIONS.HINT.TYPE && !dialogHintContainer) {
            return;
        }
        if (dialogType === CONSTANTS.DIALOG_FRAME_OPTIONS.LOCATOR.TYPE && !iframeDialogLocator) {
            return;
        }

        const iframeDialog = dialogType === CONSTANTS.DIALOG_FRAME_OPTIONS.HINT.TYPE ?
            iframeDialogHint :
            iframeDialogLocator;
        const dialogContainer = dialogType === CONSTANTS.DIALOG_FRAME_OPTIONS.HINT.TYPE ?
            dialogHintContainer :
            dialogLocatorContainer;

        iframeDialog.style.height = Math.round(dialogHeight) + 'px';
        iframeDialog.style.width = Math.round(dialogWidth) + 'px';

        const dialogContainerRect = dialogContainer.getBoundingClientRect();
        const dialogContainerHeight = dialogContainerRect.height;
        const dialogContainerWidth = dialogContainerRect.width;

        const pageHeight = document.documentElement.clientHeight;
        const pageWidth = document.documentElement.clientWidth;
        let dialogTop = 0;
        let dialogLeft = 0;
        if (originEvent && originEvent.clientX && originEvent.clientY) {
            dialogTop = originEvent.clientY;
            if (dialogTop + dialogContainerHeight > pageHeight) {
                dialogTop = Math.max(0, originEvent.clientY - dialogContainerHeight);
            }
            dialogLeft = originEvent.clientX;
            if (dialogLeft + dialogContainerWidth > pageWidth) {
                dialogLeft = Math.max(0, originEvent.clientX - dialogContainerWidth);
            }
        } else {
            dialogTop = pageHeight / 2 - dialogContainerHeight / 2;
            dialogLeft = pageWidth / 2 - dialogContainerWidth / 2;
        }
        dialogContainer.style.top = Math.round(dialogTop) + 'px';
        dialogContainer.style.left = Math.round(dialogLeft) + 'px';
        dialogContainer.style.visibility = CONSTANTS.VISIBILITY.VISIBLE;
    };

    const closeDialog = dialogType => {
        if (dialogType === CONSTANTS.DIALOG_FRAME_OPTIONS.HINT.TYPE) {
            if (!dialogHintContainer) {
                return;
            }
            if (dialogHintContainer.style.visibility === CONSTANTS.VISIBILITY.VISIBLE) {
                dialogHintContainer.style.visibility = CONSTANTS.VISIBILITY.HIDDEN;
            }
        }
        if (dialogType === CONSTANTS.DIALOG_FRAME_OPTIONS.LOCATOR.TYPE) {
            if (!dialogLocatorContainer) {
                return;
            }
            if (dialogLocatorContainer.style.visibility === CONSTANTS.VISIBILITY.VISIBLE) {
                dialogLocatorContainer.style.visibility = CONSTANTS.VISIBILITY.HIDDEN;
            }
        }
    };

    const setDialogVisible = (dialogType, isVisible) => {
        if (dialogType === CONSTANTS.DIALOG_FRAME_OPTIONS.HINT.TYPE) {
            if (!dialogHintContainer) {
                return;
            }
            dialogHintContainer.style.visibility = isVisible ? CONSTANTS.VISIBILITY.VISIBLE : CONSTANTS.VISIBILITY.HIDDEN;
        }
        if (dialogType === CONSTANTS.DIALOG_FRAME_OPTIONS.LOCATOR.TYPE) {
            if (!dialogLocatorContainer) {
                return;
            }
            dialogLocatorContainer.style.visibility = isVisible ? CONSTANTS.VISIBILITY.VISIBLE : CONSTANTS.VISIBILITY.HIDDEN;
        }
    };

    const checkIfTargetElementIsDialogFrame = id => {
        return id === dialogHintBlockId.concat(CONSTANTS.DIALOG_ELEMENTS.HANDLE.LABEL) ||
            id === dialogHintBlockId.concat(CONSTANTS.DIALOG_ELEMENTS.CLOSE_CONTAINER.LABEL) ||
            id === dialogHintBlockId.concat(CONSTANTS.DIALOG_ELEMENTS.CLOSE_BUTTON.LABEL) ||
            id === dialogLocatorBlockId.concat(CONSTANTS.DIALOG_ELEMENTS.HANDLE.LABEL) ||
            id === dialogLocatorBlockId.concat(CONSTANTS.DIALOG_ELEMENTS.CLOSE_CONTAINER.LABEL) ||
            id === dialogLocatorBlockId.concat(CONSTANTS.DIALOG_ELEMENTS.CLOSE_BUTTON.LABEL)
    };

    const checkIfTargetElementIsHtmlOrBody = target => {
        if (!target.tagName) {
            return false;
        }

        return getTagName(target) === CONSTANTS.TAGS.HTML || getTagName(target) === CONSTANTS.TAGS.BODY;
    };

    const getTagName = target => {
        return target.tagName.toLowerCase();
    };

    const showMessage = string => {
        if (navigator.webdriver) {
            chrome.runtime.sendMessage({
                action: CONSTANTS.ACTIONS.SEND_CHROME_NOTIFICATION,
                message: string
            });
        } else {
            alert(string);
        }
    };


    const getIsFramesetProperty = async window => {
        if (!window.frameElement) {
            return window.document.body.nodeName === CONSTANTS.TAGS.FRAMESET;
        }
        const checkIsFrameset = (paramWindow) => new Promise((resolve, reject) => {
            const channel = new MessageChannel();

            channel.port1.onmessage = event => {
                channel.port1.close();
                if (event.data.error) {
                    reject(event.data.error);
                } else {
                    resolve(event.data.result);
                }
            };

            paramWindow.top.postMessage(
                {
                    sender: CONSTANTS.MESSAGE_SENDERS.CHILD_WINDOW,
                    action: CONSTANTS.ACTIONS.CHECK_IS_FRAMESET
                },
                '*',
                [channel.port2]
            );
        });

        return await checkIsFrameset(window);
    };

    const generateCssForElementInShadowDom = async () => {
        try {
            return subject7Gutenberg.cssSelectorEngine.generateCssForElementInShadowDom(targetComposedPath);
        } catch (error) {
            let isFrameset = await getIsFramesetProperty(window);
            if (isFrameset) {
                showMessage(error.message);
            } else {
                showErrorNotification(error.message);
            }
            return null;
        }
    };

    const addSuccessMessageBlock = () => {
        let resultFrameAnalysis = getFrameInfo();
        if (resultFrameAnalysis.framed) {
            return;
        }

        const documentToBLock = window.frameElement ? window.top.document : window.document;

        if (documentToBLock.getElementById(successMessageBlockId)) {
            return;
        }

        successMessageBlock = documentToBLock.createElement(CONSTANTS.TAGS.IFRAME);
        successMessageBlock.setAttribute(CONSTANTS.ATTRIBUTES.ID, successMessageBlockId);
        successMessageBlock.setAttribute(CONSTANTS.ATTRIBUTES.SRC,
            chrome.runtime.getURL(CONSTANTS.FRAMES.SUCCESS_TOAST.URL));
        successMessageBlock.setAttribute(CONSTANTS.ATTRIBUTES.STYLE,
            'box-sizing: border-box; ' +
            'overflow: hidden; ' +
            'position: fixed; ' +
            'top: 10px; ' +
            'left: 50%; ' +
            'min-height: 52px; ' +
            'height: auto; ' +
            'width: 461px; ' +
            'transform: translateX(-50%); ' +
            'border: none; ' +
            'border-radius: 5px; ' +
            'background: white; ' +
            'box-shadow: 2px 4px 4px rgba(0, 0, 0, 0.25); ' +
            'visibility: hidden; ' +
            'z-index: 2147483647; '
        );
        documentToBLock.body.append(successMessageBlock);
    };

    const removeSuccessMessageBlock = () => {
        let resultFrameAnalysis = getFrameInfo();

        if (resultFrameAnalysis.framed && !resultFrameAnalysis.crossOrigin) {
            successMessageBlock = getBlockById(successMessageBlockId);
        }
        if (!successMessageBlock) {
            return;
        }

        if (!successMessageBlock.parentElement) {
            return;
        }

        successMessageBlock.parentElement.removeChild(successMessageBlock);
        successMessageBlock = undefined;
    };

    const getBlockById = id => {
        try {
            return window.top.document.getElementById(id);
        } catch (e) {
            return null;
        }
    };

    const addErrorMessageBlock = () => {
        let resultFrameAnalysis = getFrameInfo();
        if (resultFrameAnalysis.framed) {
            return;
        }

        const documentToBLock = window.frameElement ? window.top.document : window.document;

        if (documentToBLock.getElementById(errorMessageBlockId)) {
            return;
        }

        errorMessageBlock = documentToBLock.createElement(CONSTANTS.TAGS.IFRAME);
        errorMessageBlock.setAttribute(CONSTANTS.ATTRIBUTES.ID, errorMessageBlockId);
        errorMessageBlock.setAttribute(CONSTANTS.ATTRIBUTES.SRC,
            chrome.runtime.getURL(CONSTANTS.FRAMES.ERROR_TOAST.URL));
        errorMessageBlock.setAttribute(CONSTANTS.ATTRIBUTES.STYLE,
            'box-sizing: border-box; ' +
            'overflow: hidden; ' +
            'position: fixed; ' +
            'top: 10px; ' +
            'left: 50%; ' +
            'min-height: 52px; ' +
            'height: auto; ' +
            'width: 461px; ' +
            'transform: translateX(-50%); ' +
            'border: none; ' +
            'border-radius: 5px; ' +
            'background: white; ' +
            'box-shadow: 2px 4px 4px rgba(0, 0, 0, 0.25); ' +
            'visibility: hidden; ' +
            'z-index: 2147483647; '
        );
        documentToBLock.body.append(errorMessageBlock);
    };

    const removeErrorMessageBlock = () => {
        let resultFrameAnalysis = getFrameInfo();

        if (resultFrameAnalysis.framed && !resultFrameAnalysis.crossOrigin) {
            errorMessageBlock = getBlockById(errorMessageBlockId);
        }
        if (!errorMessageBlock) {
            return;
        }

        if (!errorMessageBlock.parentElement) {
            return;
        }

        errorMessageBlock.parentElement.removeChild(errorMessageBlock);
        errorMessageBlock = undefined;
    };

    const showSuccessNotification = message => {
        let resultFrameAnalysis = getFrameInfo();

        if (resultFrameAnalysis.framed && !resultFrameAnalysis.crossOrigin) {
            successMessageBlock = getBlockById(successMessageBlockId);
        }
        if (!successMessageBlock) {
            return;
        }

        if (successMessageBlock.style.visibility === CONSTANTS.VISIBILITY.VISIBLE) {
            successMessageBlock.style.visibility = CONSTANTS.VISIBILITY.HIDDEN;
        }

        successMessageBlock.contentWindow.postMessage(
            {
                sender: CONSTANTS.MESSAGE_SENDERS.S7_NOTIFICATION,
                type: CONSTANTS.NOTIFICATION_TYPES.SUCCESS,
                message
            },
            '*'
        );

        setTimeout(() => {
            hideSuccessNotification();
        }, CONSTANTS.FRAMES.SUCCESS_TOAST.DELAY);
    };

    const hideSuccessNotification = () => {
        let resultFrameAnalysis = getFrameInfo();

        if (resultFrameAnalysis.framed && !resultFrameAnalysis.crossOrigin) {
            successMessageBlock = getBlockById(successMessageBlockId);
        }
        if (!successMessageBlock) {
            return;
        }

        successMessageBlock.style.visibility = CONSTANTS.VISIBILITY.HIDDEN;
    };

    const showErrorNotification = message => {
        let resultFrameAnalysis = getFrameInfo();

        if (resultFrameAnalysis.framed && !resultFrameAnalysis.crossOrigin) {
            errorMessageBlock = getBlockById(errorMessageBlockId);
        }
        if (!errorMessageBlock) {
            return;
        }

        if (errorMessageBlock.style.visibility === CONSTANTS.VISIBILITY.VISIBLE) {
            errorMessageBlock.style.visibility = CONSTANTS.VISIBILITY.HIDDEN;
        }

        errorMessageBlock.contentWindow.postMessage(
            {
                sender: CONSTANTS.MESSAGE_SENDERS.S7_NOTIFICATION,
                type: CONSTANTS.NOTIFICATION_TYPES.ERROR,
                message
            },
            '*'
        );

        setTimeout(() => {
            hideErrorNotification();
        }, CONSTANTS.FRAMES.ERROR_TOAST.DELAY);
    };

    const hideErrorNotification = () => {
        let resultFrameAnalysis = getFrameInfo();

        if (resultFrameAnalysis.framed && !resultFrameAnalysis.crossOrigin) {
            errorMessageBlock = getBlockById(errorMessageBlockId);
        }
        if (!errorMessageBlock) {
            return;
        }

        errorMessageBlock.style.visibility = CONSTANTS.VISIBILITY.HIDDEN;
    };

    const syncNotificationSize = ({type, width, height}) => {
        if (type === CONSTANTS.NOTIFICATION_TYPES.SUCCESS && !successMessageBlock) {
            return;
        }
        if (type === CONSTANTS.NOTIFICATION_TYPES.ERROR && !errorMessageBlock) {
            return;
        }

        const iframeBlock = type === CONSTANTS.NOTIFICATION_TYPES.SUCCESS ?
            successMessageBlock :
            errorMessageBlock;

        iframeBlock.style.height = Math.round(height) + 'px';
        iframeBlock.style.width = Math.round(width) + 'px';
        iframeBlock.style.visibility = CONSTANTS.VISIBILITY.VISIBLE;
    };

    const showNotification = (message, type) => {
        switch (type) {
            case CONSTANTS.NOTIFICATION_TYPES.SUCCESS:
                showSuccessNotification(message);
                break;
            case CONSTANTS.NOTIFICATION_TYPES.ERROR:
                showErrorNotification(message);
                break;
            default:
                break;
        }
    };

    const closeNotification = type => {
        switch (type) {
            case CONSTANTS.NOTIFICATION_TYPES.SUCCESS:
                hideSuccessNotification();
                break;
            case CONSTANTS.NOTIFICATION_TYPES.ERROR:
                hideErrorNotification();
                break;
            default:
                break;
        }
    };

    const getTargetElement = () => {
        return target;
    };

    const getHintValue = (finalHint, initialHint) => {
        let hasHint = !!initialHint;
        let isWildCard = hasHint && initialHint.startsWith('*');

        return !hasHint ?
            finalHint ?
                `used empty hint, value was found during generation: "${finalHint}"` :
                'used empty hint, value was not found during generation' :
            isWildCard ?
                finalHint ?
                    `used hint with wildcard "${initialHint}", value was found during generation: "${finalHint}"` :
                    'used hint with wildcard "${initialHint}", value was not found during generation' :
                (initialHint === finalHint) ?
                    `"${initialHint}"` :
                    `used hint "${initialHint}", value was found during generation: "${finalHint}"`;
    };

    const getAlgorithmType = info => {
        let algorithmType = CONSTANTS.ALGORITHM_TYPES.ABSOLUTE_XPATH;

        if (info.patternId) {
            algorithmType = CONSTANTS.ALGORITHM_TYPES.TEMPLATES;
        } else if (info.errors.length === 1 &&
            (info.errors[0].includes(CONSTANTS.ERROR_MESSAGES.MISSING_PATTERNS) ||
                info.errors[0].includes(CONSTANTS.ERROR_MESSAGES.NO_MATCHING_TEMPLATE))) {
            algorithmType = CONSTANTS.ALGORITHM_TYPES.SMART_XPATH;
        }

        return algorithmType;
    };

    return {
        generateXpath: generateXpath,
        generateCssForElementInShadowDom: generateCssForElementInShadowDom,
        resetCapturing: resetCapturing,
        startCapturing: startCapturing,
        stopCapturing: stopCapturing,
        test: test,
        openDialog: openDialog,
        closeDialog: closeDialog,
        syncDialogSize: syncDialogSize,
        setDialogVisible: setDialogVisible,
        getIsFramesetProperty: getIsFramesetProperty,
        getFrameInfo: getFrameInfo,
        isSVGElement: isSVGElement,
        isInShadowRoot: isInShadowRoot,
        showNotification: showNotification,
        syncNotificationSize: syncNotificationSize,
        closeNotification: closeNotification,
        testCssSelectorAfterGeneration: testCssSelectorAfterGeneration,
        getTargetElement: getTargetElement,
        getHintValue: getHintValue,
        getAlgorithmType: getAlgorithmType,
        containsCustomNamespace: containsCustomNamespace,
        getTagName: getTagName
    };

})();

chrome.runtime.onMessage.addListener(async (request, sender, sendResponse) => {
    let action = request.action;

    if (action === CONSTANTS.ACTIONS.CHECK_IS_FRAMESET) {
        const isFrameset = await subject7Gutenberg.locatorRecorder.getIsFramesetProperty(window);
        sendResponse({isFrameset: isFrameset});
        return true;
    }

    switch (action) {
        case CONSTANTS.ACTIONS.START_CAPTURING:
            subject7Gutenberg.locatorRecorder.startCapturing();
            sendResponse({started: started});
            break;
        case CONSTANTS.ACTIONS.STOP_CAPTURING:
            subject7Gutenberg.locatorRecorder.stopCapturing();
            sendResponse({started: started});
            break;
        case CONSTANTS.ACTIONS.TEST_XPATH:
            if (request.xpath) {
                await subject7Gutenberg.locatorRecorder.test(request.xpath, CONSTANTS.TEST_TYPES.XPATH, () => {
                    chrome.runtime.sendMessage({
                        action: CONSTANTS.ACTIONS.TEST_XPATH_FINISHED,
                        css: request.css,
                        isFrameset: request.isFrameset
                    });
                });
            } else {
                chrome.runtime.sendMessage({
                    action: CONSTANTS.ACTIONS.TEST_XPATH_FINISHED,
                    css: request.css,
                    isFrameset: request.isFrameset
                });
            }
            break;
        case CONSTANTS.ACTIONS.TEST_CSS:
            if (request.css) {
                await subject7Gutenberg.locatorRecorder.test(request.css, CONSTANTS.TEST_TYPES.CSS, () => {
                    chrome.runtime.sendMessage({
                        action: CONSTANTS.ACTIONS.TEST_CSS_FINISHED,
                        isFrameset: request.isFrameset
                    });
                });
            } else {
                chrome.runtime.sendMessage({
                    action: CONSTANTS.ACTIONS.TEST_CSS_FINISHED,
                    isFrameset: request.isFrameset
                });
            }
            break;
        case CONSTANTS.ACTIONS.GENERATE_XPATH:
            const {hint, isDisabledHint} = request;
            let result = await subject7Gutenberg.locatorRecorder.generateXpath(hint);

            if (result) {
                const isFrameset = await subject7Gutenberg.locatorRecorder.getIsFramesetProperty(window);
                if (result.xpath) {
                    let targetElement = subject7Gutenberg.locatorRecorder.getTargetElement();
                    let hintValue = subject7Gutenberg.locatorRecorder.getHintValue(result.hint, hint.text);
                    let algorithmType = subject7Gutenberg.locatorRecorder.getAlgorithmType(result);
                    result.algorithmType = algorithmType;
                    chrome.runtime.sendMessage({
                        action: CONSTANTS.ACTIONS.SEND_LOG_INFO,
                        logInfo: result.logInfo,
                        element: targetElement.outerHTML,
                        hint: hintValue,
                        additionalInfo: {
                            errors: result.errors,
                            algorithmType: algorithmType,
                            xpath: result.xpath,
                            patternDate: result.patternDate
                        }
                    });
                    console.debug(`${new Date(Date.now()).toISOString()} -> [XPATH_GENERATION] Element "`, targetElement,
                        `", Hint info: ${hintValue}. Log Info about XPath templates:`, result.logInfo);
                    console.debug(`${new Date(Date.now()).toISOString()} -> [XPATH_GENERATION] XPath "${result.xpath}" generated using ${algorithmType}`);

                    try {
                        result.css = await subject7Gutenberg.cssSelectorEngine.xPathToCss(result.xpath);
                    } catch (error) {}
                } else {
                    result.css = await subject7Gutenberg.locatorRecorder.generateCssForElementInShadowDom();
                }
                chrome.runtime.sendMessage({
                    action: CONSTANTS.ACTIONS.GENERATION_FINISHED,
                    result,
                    hint,
                    isFrameset,
                    isDisabledHint
                });
            }
            break;
        case CONSTANTS.ACTIONS.RESET_CAPTURING:
            subject7Gutenberg.locatorRecorder.resetCapturing(request.fireEvent);
            break;
        case CONSTANTS.ACTIONS.OPEN_DIALOG:
            const {dialogType, options} = request;
            subject7Gutenberg.locatorRecorder.openDialog(dialogType, options);
            break;
        case CONSTANTS.ACTIONS.CLOSE_DIALOG:
            subject7Gutenberg.locatorRecorder.closeDialog(request.dialogType);
            break;
        case CONSTANTS.ACTIONS.SYNC_DIALOG_SIZE:
            subject7Gutenberg.locatorRecorder.syncDialogSize(request);
            break;
        case CONSTANTS.ACTIONS.HIDE_DIALOG:
            subject7Gutenberg.locatorRecorder.setDialogVisible(request.dialogType, false);
            break;
        case CONSTANTS.ACTIONS.SHOW_DIALOG:
            subject7Gutenberg.locatorRecorder.setDialogVisible(request.dialogType, true);
            break;
        case CONSTANTS.ACTIONS.SHOW_ALERT:
            if (window === window.top) {
                alert(request.message);
            }
            break;
        case CONSTANTS.ACTIONS.SHOW_NOTIFICATION:
            if (window === window.top) {
                subject7Gutenberg.locatorRecorder.showNotification(request.message, request.type);
            }
            break;
        case CONSTANTS.ACTIONS.SYNC_NOTIFICATION_SIZE:
            subject7Gutenberg.locatorRecorder.syncNotificationSize(request);
            break;
        case CONSTANTS.ACTIONS.CLOSE_NOTIFICATION:
            subject7Gutenberg.locatorRecorder.closeNotification(request.type);
            break;
        default:
            console.debug(`${new Date(Date.now()).toISOString()} -> Action is not defined: ${JSON.stringify(request)}`);
            break;
    }
    sendResponse();
    return true;
});

let port;
let isStarted = false;
const connect = () => {
    port = chrome.runtime.connect(chrome.runtime.id, {name: CONSTANTS.MESSAGING_PORT});
    if (!isStarted) {
        console.debug(`${new Date(Date.now()).toISOString()} -> [KEEP_ALIVE] Content script with id=${uuid} starts connection to port '${port.name}'`);
        isStarted = true;
    }
    port.onDisconnect.addListener(connect);
    port.onMessage.addListener(msg => {
    });
    port.postMessage({uuid: uuid, isAutomationBrowser: navigator.webdriver});
};

if (!subject7Gutenberg.locatorRecorder.getFrameInfo().framed) {
    connect();
}
